function f2_extract_bodies_from_SE(shape_paths, image_size)

% Description.
% - Finds lists of SE files; extracts Cartesian coordinates; saves them (f1_read_se_shape_file)
% - Converts Cartesian coordinates into binary Matrix (f2_generate_binary_from_coordinates)
% - Saves binary Matrices as images 
%
% Dmitry Ershov 
% 2017 October



%% PARAMETERS:
params.image_size               = image_size;       % size of the output image
params.image_digit_format       = 4;                % number of digits in the image file name; padded with 0; i.e.: 'shape0015.TIF'
params.image_format             = 'TIF';            % image format

params.image_open_cortex_by     = 3;                % size of the structuring element to close small holes (they appear if the mesh is not dense enough); 




%% EXTRACT COORDINATES FROM SURFACE EVOLVER FILES 

% try to load the extracted data (if it exists). 
file_name           = 'shape_data.mat';
file_path           = [shape_paths.shapes_source, file_name];

if ~exist(file_path, 'file')    
    % if the data does not exist: extract it.
    
    % Extracting coordinates from the shape files in a form of the 2D double matrix
    [vertices_mat, edges_mat, faces_mat, bodies_mat, errcheck] = f3_read_se_shape_file(shape_paths);
    
    % warnings
    if errcheck == 1
        disp(' ')
        disp(['There were NaNs in some matrices; they were removed.']);
    end
    if isempty(vertices_mat)
        disp(' ')
        disp(['Matrices are empty. Quitting.']);   
        return;
    end

    disp(' ')
    disp(['Saving all shape matrices to: ']);
    disp(['    ', file_path]);
    save(file_path, 'bodies_mat', 'faces_mat', 'edges_mat', 'vertices_mat');
    disp(['    Done']);

else
    % otherwise extract and save
    disp(' ')
    disp(['Found existing vertex data; loading files: ']);
    disp(['    ', file_path]);
    load(file_path, 'vertices_mat', 'edges_mat', 'faces_mat', 'bodies_mat'); 
    disp(['    ', 'To extract anew, delete  manually the file: ', file_name]);
end






%%  LABEL THE BODIES AND SAVE THEM AS UINT8 MATRIX 

% try to load the labeled bodies data (if it exists). 
file_name           = 'labeled_bodies.mat';
file_path           = [shape_paths.shapes_source, file_name];

if ~exist(file_path, 'file')    

    % if the data does not exist: label the bodies.

    % 1. Make a binary 0-1 matrix for each existing body.
    % go over each body -> over each face in it -> map points from Cartesian space to Image size-specified space.
    % by convention:  0 = interior of a body; 1 = exterior of a body
    all_bodies_labeled_uint8 = f4_generate_binary_from_coordinates(vertices_mat, edges_mat, faces_mat, bodies_mat, params);


    disp(' ')
    disp(['Saving the matrix with labeled bodies: ']);
    disp(['    ', file_path]);
    save(file_path, 'all_bodies_labeled_uint8');
    disp(['    Done']);

else
    disp(' ')
    disp(['Found existing labeled bodies data; loading files: ']);
    disp(['    ', file_path]);
    disp(['    ', 'To extract anew, delete manually the file: ', file_name]);

end





